<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\RequestException;
use Carbon\Carbon;

class MootaService
{
    /**
     * Moota API V2 Base URL
     */
    private const BASE_URL = 'https://app.moota.co/api/v2';

    /**
     * Cache TTL defaults (in seconds)
     */
    private const CACHE_BANK_LIST_TTL = 3600;       // 1 hour
    private const CACHE_MUTATION_TTL  = 60;          // 1 minute (mutations change frequently)

    /**
     * Retry configuration
     */
    private const MAX_RETRIES   = 3;
    private const RETRY_DELAY   = 500; // ms

    private string $apiToken;
    private int $timeout;

    public function __construct()
    {
        $this->apiToken = config('services.moota.api_token');
        $this->timeout  = config('services.moota.timeout', 30);

        if (empty($this->apiToken)) {
            throw new \RuntimeException('Moota API token is not configured. Set MOOTA_API_TOKEN in your .env file.');
        }
    }

    // =========================================================================
    // HTTP CLIENT
    // =========================================================================

    /**
     * Build a configured HTTP client instance.
     */
    private function client(): PendingRequest
    {
        return Http::baseUrl(self::BASE_URL)
            ->withToken($this->apiToken)
            ->acceptJson()
            ->timeout($this->timeout)
            ->retry(self::MAX_RETRIES, self::RETRY_DELAY, function (\Exception $e, PendingRequest $request) {
                // Only retry on server errors or timeout, not on 4xx
                if ($e instanceof RequestException) {
                    return $e->response?->status() >= 500;
                }
                return true; // retry on connection errors
            });
    }

    /**
     * Execute GET request with error handling.
     */
    private function get(string $endpoint, array $params = []): array
    {
        try {
            $response = $this->client()->get($endpoint, $params);

            if ($response->failed()) {
                $this->handleError('GET', $endpoint, $response);
            }

            return $response->json() ?? [];
        } catch (RequestException $e) {
            $this->logError('GET', $endpoint, $e);
            throw $e;
        }
    }

    /**
     * Execute POST request with error handling.
     */
    private function post(string $endpoint, array $data = []): array
    {
        try {
            $response = $this->client()->post($endpoint, $data);

            if ($response->failed()) {
                $this->handleError('POST', $endpoint, $response);
            }

            return $response->json() ?? [];
        } catch (RequestException $e) {
            $this->logError('POST', $endpoint, $e);
            throw $e;
        }
    }

    // =========================================================================
    // BANK ACCOUNTS
    // =========================================================================

    /**
     * Get list of registered bank accounts.
     *
     * @param  int  $page
     * @param  int  $perPage
     * @param  bool $useCache
     * @return array
     */
    public function getBankAccounts(int $page = 1, int $perPage = 20, bool $useCache = true): array
    {
        $cacheKey = "moota:banks:page_{$page}_per_{$perPage}";

        if ($useCache && Cache::has($cacheKey)) {
            return Cache::get($cacheKey);
        }

        $result = $this->get('/bank', [
            'page'     => $page,
            'per_page' => $perPage,
        ]);

        if ($useCache) {
            Cache::put($cacheKey, $result, self::CACHE_BANK_LIST_TTL);
        }

        return $result;
    }

    /**
     * Get all bank accounts (auto-paginate).
     *
     * @return array Collection of all bank account data
     */
    public function getAllBankAccounts(): array
    {
        $cacheKey = 'moota:banks:all';

        if (Cache::has($cacheKey)) {
            return Cache::get($cacheKey);
        }

        $allBanks = [];
        $page = 1;

        do {
            $response = $this->getBankAccounts($page, 50, false);
            $data     = $response['data'] ?? [];
            $allBanks = array_merge($allBanks, $data);
            $lastPage = $response['last_page'] ?? 1;
            $page++;
        } while ($page <= $lastPage);

        Cache::put($cacheKey, $allBanks, self::CACHE_BANK_LIST_TTL);

        return $allBanks;
    }

    /**
     * Find a specific bank account by bank_id.
     *
     * @param  string $bankId
     * @return array|null
     */
    public function findBank(string $bankId): ?array
    {
        $banks = $this->getAllBankAccounts();

        return collect($banks)->firstWhere('bank_id', $bankId);
    }

    /**
     * Refresh mutation data for a specific bank (triggers immediate scraping).
     * Note: This costs Moota points.
     *
     * @param  string $bankId
     * @return array
     */
    public function refreshBankMutation(string $bankId): array
    {
        return $this->post("/bank/{$bankId}/refresh");
    }

    // =========================================================================
    // MUTATIONS
    // =========================================================================

    /**
     * Get mutations with flexible filters.
     *
     * @param  array $filters Supported keys:
     *   - type       : 'CR' (credit) or 'DB' (debit)
     *   - bank       : bank_id to filter by specific bank
     *   - amount     : filter by specific amount
     *   - note       : filter by note content
     *   - start_date : Y-m-d format
     *   - end_date   : Y-m-d format
     *   - tag        : comma-separated tag names
     *   - page       : page number
     *   - per_page   : results per page
     * @return array
     */
    public function getMutations(array $filters = []): array
    {
        $params = array_filter([
            'type'       => $filters['type'] ?? null,
            'bank'       => $filters['bank'] ?? null,
            'amount'     => $filters['amount'] ?? null,
            'note'       => $filters['note'] ?? null,
            'start_date' => $filters['start_date'] ?? null,
            'end_date'   => $filters['end_date'] ?? null,
            'tag'        => $filters['tag'] ?? null,
            'page'       => $filters['page'] ?? 1,
            'per_page'   => $filters['per_page'] ?? 20,
        ], fn($v) => $v !== null);

        return $this->get('/mutation', $params);
    }

    /**
     * Get CREDIT (incoming) mutations only.
     */
    public function getCreditMutations(array $filters = []): array
    {
        return $this->getMutations(array_merge($filters, ['type' => 'CR']));
    }

    /**
     * Get DEBIT (outgoing) mutations only.
     */
    public function getDebitMutations(array $filters = []): array
    {
        return $this->getMutations(array_merge($filters, ['type' => 'DB']));
    }

    /**
     * Add a note to a mutation.
     *
     * @param  string $mutationId
     * @param  string $note
     * @return array
     */
    public function addMutationNote(string $mutationId, string $note): array
    {
        return $this->post("/mutation/{$mutationId}/note", [
            'note' => $note,
        ]);
    }

    /**
     * Push/retry webhook for a specific mutation.
     *
     * @param  string $mutationId
     * @return array
     */
    public function pushMutationWebhook(string $mutationId): array
    {
        return $this->post("/mutation/{$mutationId}/webhook");
    }

    /**
     * Delete mutations by IDs.
     *
     * @param  array $mutationIds
     * @return array
     */
    public function destroyMutations(array $mutationIds): array
    {
        return $this->post('/mutation/destroy', [
            'mutations' => $mutationIds,
        ]);
    }

    /**
     * Create a dummy/test mutation (for development only).
     *
     * @param  string $bankId
     * @param  array  $data  Keys: date, note, amount, type
     * @return array
     */
    public function storeDummyMutation(string $bankId, array $data): array
    {
        return $this->post("/mutation/store/{$bankId}", [
            'date'   => $data['date'] ?? now()->format('Y-m-d'),
            'note'   => $data['note'] ?? '',
            'amount' => $data['amount'] ?? 0,
            'type'   => $data['type'] ?? 'CR',
        ]);
    }

    // =========================================================================
    // TAGGING
    // =========================================================================

    /**
     * Attach tags to a mutation.
     */
    public function attachTag(string $mutationId, array $tagNames): array
    {
        return $this->post("/tagging/mutation/{$mutationId}", [
            'name' => $tagNames,
        ]);
    }

    // =========================================================================
    // PAYMENT VERIFICATION
    // =========================================================================

    /**
     * Verify a payment by matching amount within a date range.
     *
     * Strategy: Search credit mutations for a matching amount.
     * Supports unique code matching (e.g., Rp 150.321 where 321 is unique code).
     *
     * @param  float       $amount         Expected total amount (including unique code)
     * @param  string|null $bankId         Filter by specific bank (optional)
     * @param  string|null $startDate      Search start date Y-m-d (default: today - 1 day)
     * @param  string|null $endDate        Search end date Y-m-d (default: today)
     * @param  string|null $description    Optional description keyword to match
     * @return array{verified: bool, mutation: ?array, mutations: array}
     */
    public function verifyPayment(
        float   $amount,
        ?string $bankId = null,
        ?string $startDate = null,
        ?string $endDate = null,
        ?string $description = null,
    ): array {
        $startDate = $startDate ?? Carbon::now()->subDay()->format('Y-m-d');
        $endDate   = $endDate ?? Carbon::now()->format('Y-m-d');

        $filters = [
            'type'       => 'CR',
            'start_date' => $startDate,
            'end_date'   => $endDate,
            'per_page'   => 100,
        ];

        if ($bankId) {
            $filters['bank'] = $bankId;
        }

        $response  = $this->getMutations($filters);
        $mutations = $response['data'] ?? [];

        // Exact amount matching
        $matched = collect($mutations)->filter(function ($mutation) use ($amount, $description) {
            $mutationAmount = (float) $mutation['amount'];
            $amountMatch    = abs($mutationAmount - $amount) < 0.01;

            if ($description) {
                $descMatch = str_contains(
                    strtolower($mutation['description'] ?? ''),
                    strtolower($description)
                );
                return $amountMatch && $descMatch;
            }

            return $amountMatch;
        });

        $firstMatch = $matched->first();

        return [
            'verified'  => $firstMatch !== null,
            'mutation'  => $firstMatch,
            'mutations' => $matched->values()->all(),
            'searched'  => [
                'amount'     => $amount,
                'bank_id'    => $bankId,
                'start_date' => $startDate,
                'end_date'   => $endDate,
                'total_cr'   => count($mutations),
            ],
        ];
    }

    /**
     * Verify payment with tolerance range (useful for unique code matching).
     *
     * Example: Base price Rp 150.000, unique code range 001-999
     *          Search for mutations between 150.001 and 150.999
     *
     * @param  float       $baseAmount   Base payment amount
     * @param  int         $uniqueCode   The expected unique code suffix
     * @param  string|null $bankId
     * @param  string|null $startDate
     * @param  string|null $endDate
     * @return array{verified: bool, mutation: ?array}
     */
    public function verifyPaymentWithUniqueCode(
        float   $baseAmount,
        int     $uniqueCode,
        ?string $bankId = null,
        ?string $startDate = null,
        ?string $endDate = null,
    ): array {
        $expectedAmount = $baseAmount + $uniqueCode;

        return $this->verifyPayment($expectedAmount, $bankId, $startDate, $endDate);
    }

    /**
     * Verify payment with amount tolerance/range (for slight variations).
     *
     * @param  float       $expectedAmount
     * @param  float       $tolerance       Acceptable difference (default: 1.00)
     * @param  string|null $bankId
     * @param  string|null $startDate
     * @param  string|null $endDate
     * @return array{verified: bool, mutation: ?array, mutations: array}
     */
    public function verifyPaymentWithTolerance(
        float   $expectedAmount,
        float   $tolerance = 1.00,
        ?string $bankId = null,
        ?string $startDate = null,
        ?string $endDate = null,
    ): array {
        $startDate = $startDate ?? Carbon::now()->subDay()->format('Y-m-d');
        $endDate   = $endDate ?? Carbon::now()->format('Y-m-d');

        $filters = [
            'type'       => 'CR',
            'start_date' => $startDate,
            'end_date'   => $endDate,
            'per_page'   => 100,
        ];

        if ($bankId) {
            $filters['bank'] = $bankId;
        }

        $response  = $this->getMutations($filters);
        $mutations = $response['data'] ?? [];

        $matched = collect($mutations)->filter(function ($mutation) use ($expectedAmount, $tolerance) {
            $diff = abs((float) $mutation['amount'] - $expectedAmount);
            return $diff <= $tolerance;
        })->sortBy(function ($mutation) use ($expectedAmount) {
            return abs((float) $mutation['amount'] - $expectedAmount);
        });

        $firstMatch = $matched->first();

        return [
            'verified'  => $firstMatch !== null,
            'mutation'  => $firstMatch,
            'mutations' => $matched->values()->all(),
            'searched'  => [
                'expected_amount' => $expectedAmount,
                'tolerance'       => $tolerance,
                'bank_id'         => $bankId,
                'start_date'      => $startDate,
                'end_date'        => $endDate,
                'total_cr'        => count($mutations),
            ],
        ];
    }

    // =========================================================================
    // WEBHOOK HANDLING
    // =========================================================================

    /**
     * Validate and parse incoming Moota webhook payload.
     *
     * Usage in your controller:
     *   $mutations = $mootaService->parseWebhookPayload($request->all());
     *
     * @param  array       $payload     Raw webhook request data
     * @param  string|null $secretToken Your webhook secret token for signature verification
     * @return array Parsed mutation data
     */
    public function parseWebhookPayload(array $payload, ?string $secretToken = null): array
    {
        // Moota webhook sends an array of mutations
        // Each mutation has: bank_id, account_number, bank_type, date, amount, description, type, balance, etc.
        $mutations = [];

        foreach ($payload as $item) {
            if (!is_array($item)) {
                continue;
            }

            $mutations[] = [
                'bank_id'        => $item['bank_id'] ?? null,
                'account_number' => $item['account_number'] ?? null,
                'bank_type'      => $item['bank_type'] ?? null,
                'date'           => $item['date'] ?? null,
                'amount'         => (float) ($item['amount'] ?? 0),
                'description'    => $item['description'] ?? '',
                'type'           => $item['type'] ?? null, // CR or DB
                'balance'        => (float) ($item['balance'] ?? 0),
                'mutation_id'    => $item['mutation_id'] ?? $item['token'] ?? null,
                'note'           => $item['note'] ?? '',
                'created_at'     => $item['created_at'] ?? null,
                'updated_at'     => $item['updated_at'] ?? null,
                'tag'            => $item['taggings'] ?? [],
            ];
        }

        return $mutations;
    }

    /**
     * Process incoming webhook and auto-verify against pending payments.
     *
     * This is a template — customize the callback to match your payment model.
     *
     * @param  array    $payload  Raw webhook data
     * @param  callable $onMatch  Callback: fn(array $mutation, $pendingPayment) => void
     * @param  callable $findPending Callback to find pending payment: fn(float $amount) => ?Model
     * @return array{processed: int, matched: int, details: array}
     */
    public function processWebhook(array $payload, callable $findPending, callable $onMatch): array
    {
        $mutations = $this->parseWebhookPayload($payload);
        $processed = 0;
        $matched   = 0;
        $details   = [];

        foreach ($mutations as $mutation) {
            $processed++;

            // Only process credit (incoming) mutations
            if (($mutation['type'] ?? '') !== 'CR') {
                $details[] = [
                    'mutation_id' => $mutation['mutation_id'],
                    'status'      => 'skipped',
                    'reason'      => 'Not a credit mutation',
                ];
                continue;
            }

            $amount  = $mutation['amount'];
            $pending = $findPending($amount);

            if ($pending) {
                $matched++;
                $onMatch($mutation, $pending);
                $details[] = [
                    'mutation_id' => $mutation['mutation_id'],
                    'amount'      => $amount,
                    'status'      => 'matched',
                ];
            } else {
                $details[] = [
                    'mutation_id' => $mutation['mutation_id'],
                    'amount'      => $amount,
                    'status'      => 'unmatched',
                ];
            }
        }

        Log::info('Moota webhook processed', [
            'total'   => $processed,
            'matched' => $matched,
        ]);

        return [
            'processed' => $processed,
            'matched'   => $matched,
            'details'   => $details,
        ];
    }

    // =========================================================================
    // ERROR HANDLING
    // =========================================================================

    /**
     * Handle HTTP error response.
     */
    private function handleError(string $method, string $endpoint, $response): void
    {
        $status = $response->status();
        $body   = $response->json() ?? $response->body();

        Log::error("Moota API Error [{$method} {$endpoint}]", [
            'status' => $status,
            'body'   => $body,
        ]);

        $message = match (true) {
            $status === 401 => 'Moota API authentication failed. Check your API token.',
            $status === 404 => 'Moota API resource not found.',
            $status === 422 => 'Moota API validation error: ' . json_encode($body['errors'] ?? $body),
            $status === 429 => 'Moota API rate limit exceeded. Try again later.',
            $status >= 500  => 'Moota API server error. Please try again later.',
            default         => "Moota API error (HTTP {$status})",
        };

        throw new \RuntimeException($message, $status);
    }

    /**
     * Log request exception.
     */
    private function logError(string $method, string $endpoint, \Exception $e): void
    {
        Log::error("Moota API Exception [{$method} {$endpoint}]", [
            'message' => $e->getMessage(),
            'code'    => $e->getCode(),
        ]);
    }

    // =========================================================================
    // UTILITY HELPERS
    // =========================================================================

    /**
     * Clear all Moota-related cache.
     */
    public function clearCache(): void
    {
        Cache::forget('moota:banks:all');
        // Clear paginated bank caches
        for ($i = 1; $i <= 10; $i++) {
            Cache::forget("moota:banks:page_{$i}_per_20");
            Cache::forget("moota:banks:page_{$i}_per_50");
        }
    }

    /**
     * Get mutation summary for a date range.
     *
     * @param  string      $startDate
     * @param  string      $endDate
     * @param  string|null $bankId
     * @return array{total_credit: float, total_debit: float, count_credit: int, count_debit: int, net: float}
     */
    public function getMutationSummary(string $startDate, string $endDate, ?string $bankId = null): array
    {
        $filters = [
            'start_date' => $startDate,
            'end_date'   => $endDate,
            'per_page'   => 100,
        ];

        if ($bankId) {
            $filters['bank'] = $bankId;
        }

        $allMutations = [];
        $page = 1;

        do {
            $filters['page'] = $page;
            $response = $this->getMutations($filters);
            $data     = $response['data'] ?? [];
            $allMutations = array_merge($allMutations, $data);
            $lastPage = $response['last_page'] ?? 1;
            $page++;
        } while ($page <= $lastPage && $page <= 50); // safety limit

        $mutations    = collect($allMutations);
        $totalCredit  = $mutations->where('type', 'CR')->sum(fn($m) => (float) $m['amount']);
        $totalDebit   = $mutations->where('type', 'DB')->sum(fn($m) => (float) $m['amount']);

        return [
            'total_credit' => $totalCredit,
            'total_debit'  => $totalDebit,
            'count_credit' => $mutations->where('type', 'CR')->count(),
            'count_debit'  => $mutations->where('type', 'DB')->count(),
            'net'          => $totalCredit - $totalDebit,
            'period'       => ['start' => $startDate, 'end' => $endDate],
        ];
    }

    /**
     * Health check — verify API connectivity and token validity.
     *
     * @return array{ok: bool, message: string, bank_count: int}
     */
    public function healthCheck(): array
    {
        try {
            $result = $this->getBankAccounts(1, 1, false);

            return [
                'ok'         => true,
                'message'    => 'Moota API connection successful.',
                'bank_count' => $result['total'] ?? 0,
            ];
        } catch (\Exception $e) {
            return [
                'ok'      => false,
                'message' => 'Moota API connection failed: ' . $e->getMessage(),
                'bank_count' => 0,
            ];
        }
    }
}
