<?php

namespace App\Http\Controllers;

use App\Services\MootaService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class MootaWebhookController extends Controller
{
    public function __construct(
        private MootaService $moota,
    ) {}

    /**
     * Handle incoming Moota webhook.
     * Route: POST /api/webhooks/moota
     *
     * Register this route in routes/api.php:
     *   Route::post('webhooks/moota', [MootaWebhookController::class, 'handle']);
     *
     * Don't forget to exclude this route from CSRF verification in
     * app/Http/Middleware/VerifyCsrfToken.php:
     *   protected $except = ['api/webhooks/moota'];
     */
    public function handle(Request $request): JsonResponse
    {
        Log::info('Moota webhook received', [
            'ip'      => $request->ip(),
            'payload' => $request->all(),
        ]);

        // Optional: Verify webhook signature
        // $signature = $request->header('X-Moota-Signature');
        // if (!$this->verifySignature($request->getContent(), $signature)) {
        //     return response()->json(['message' => 'Invalid signature'], 403);
        // }

        $result = $this->moota->processWebhook(
            payload: $request->all(),

            // Find pending payment by amount — customize this to match your model
            findPending: function (float $amount) {
                // Example with a Payment model:
                // return \App\Models\Payment::where('status', 'pending')
                //     ->where('total_amount', $amount)
                //     ->where('expired_at', '>', now())
                //     ->orderBy('created_at', 'desc')
                //     ->first();

                return null; // Replace with your implementation
            },

            // Handle matched payment — customize this
            onMatch: function (array $mutation, $pendingPayment) {
                // Example:
                // $pendingPayment->update([
                //     'status'          => 'paid',
                //     'paid_at'         => now(),
                //     'mutation_id'     => $mutation['mutation_id'],
                //     'payment_proof'   => json_encode($mutation),
                // ]);
                //
                // event(new \App\Events\PaymentVerified($pendingPayment));

                Log::info('Payment matched', [
                    'mutation_id' => $mutation['mutation_id'],
                    'amount'      => $mutation['amount'],
                ]);
            },
        );

        return response()->json([
            'message'   => 'Webhook processed',
            'processed' => $result['processed'],
            'matched'   => $result['matched'],
        ]);
    }
}


// =============================================================================
// USAGE EXAMPLES
// =============================================================================
//
// 1. MANUAL PAYMENT VERIFICATION (polling approach)
// --------------------------------------------------
//
//   $moota = app(MootaService::class);
//
//   // Verify exact amount
//   $result = $moota->verifyPayment(
//       amount:    150321,          // Rp 150.321
//       bankId:    'VLagzqBj42D',  // optional: specific bank
//       startDate: '2025-02-01',
//       endDate:   '2025-02-09',
//   );
//
//   if ($result['verified']) {
//       $mutation = $result['mutation'];
//       // Mark payment as paid...
//   }
//
//
// 2. UNIQUE CODE VERIFICATION
// ----------------------------
//
//   // Base: Rp 150.000, Unique code: 321
//   $result = $moota->verifyPaymentWithUniqueCode(
//       baseAmount: 150000,
//       uniqueCode: 321,
//       bankId:     'VLagzqBj42D',
//   );
//
//
// 3. TOLERANCE-BASED VERIFICATION
// --------------------------------
//
//   // Accept Rp 149.999 - Rp 150.001
//   $result = $moota->verifyPaymentWithTolerance(
//       expectedAmount: 150000,
//       tolerance:      1.00,
//   );
//
//
// 4. LIST BANK ACCOUNTS
// ----------------------
//
//   $banks = $moota->getAllBankAccounts();
//   foreach ($banks as $bank) {
//       echo "{$bank['label']} - {$bank['account_number']} ({$bank['atas_nama']})\n";
//   }
//
//
// 5. GET RECENT CREDIT MUTATIONS
// --------------------------------
//
//   $mutations = $moota->getCreditMutations([
//       'start_date' => '2025-02-01',
//       'end_date'   => '2025-02-09',
//       'bank'       => 'VLagzqBj42D',
//   ]);
//
//
// 6. MUTATION SUMMARY
// --------------------
//
//   $summary = $moota->getMutationSummary('2025-02-01', '2025-02-28', 'VLagzqBj42D');
//   // Returns: total_credit, total_debit, count_credit, count_debit, net
//
//
// 7. HEALTH CHECK
// ----------------
//
//   $status = $moota->healthCheck();
//   // ['ok' => true, 'message' => 'Moota API connection successful.', 'bank_count' => 3]
//
//
// 8. ARTISAN COMMAND FOR CRON-BASED VERIFICATION (scheduled polling)
// -------------------------------------------------------------------
//
//   // In App\Console\Commands\VerifyPendingPayments.php
//   //
//   // Schedule: $schedule->command('payments:verify')->everyFiveMinutes();
//   //
//   // public function handle(MootaService $moota) {
//   //     $pending = Payment::where('status', 'pending')
//   //         ->where('expired_at', '>', now())
//   //         ->get();
//   //
//   //     foreach ($pending as $payment) {
//   //         $result = $moota->verifyPayment(
//   //             amount:    $payment->total_amount,
//   //             startDate: $payment->created_at->format('Y-m-d'),
//   //         );
//   //
//   //         if ($result['verified']) {
//   //             $payment->update([
//   //                 'status'      => 'paid',
//   //                 'paid_at'     => now(),
//   //                 'mutation_id' => $result['mutation']['mutation_id'],
//   //             ]);
//   //         }
//   //     }
//   // }
//
