<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use App\Services\MootaService;
use Illuminate\Http\Request;

class BankController extends Controller
{
    public function __construct(
        private MootaService $mootaService,
    ) {
    }

    public function index(Request $request)
    {
        $query = Bank::query();

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('bank_name', 'like', "%{$request->search}%")
                    ->orWhere('account_name', 'like', "%{$request->search}%")
                    ->orWhere('account_number', 'like', "%{$request->search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        $banks = $query->latest()->paginate(15);

        return view('admin.banks.index', compact('banks'));
    }

    public function create()
    {
        // Fetch banks from Moota for selection
        $mootaBanks = [];
        try {
            $mootaBanks = $this->mootaService->getAllBankAccounts();
        } catch (\Exception $e) {
            // Silent fail, user can enter manually
        }

        return view('admin.banks.create', compact('mootaBanks'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'moota_bank_id' => 'required|string|unique:banks,moota_bank_id',
            'bank_name' => 'required|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_number' => 'required|string|max:50',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);

        Bank::create($validated);

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank berhasil ditambahkan.');
    }

    public function edit(Bank $bank)
    {
        return view('admin.banks.edit', compact('bank'));
    }

    public function update(Request $request, Bank $bank)
    {
        $validated = $request->validate([
            'moota_bank_id' => 'required|string|unique:banks,moota_bank_id,' . $bank->id,
            'bank_name' => 'required|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_number' => 'required|string|max:50',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);

        $bank->update($validated);

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank berhasil diperbarui.');
    }

    public function destroy(Bank $bank)
    {
        // Check if bank has orders
        if ($bank->orders()->exists()) {
            return back()->with('error', 'Tidak dapat menghapus bank yang memiliki order.');
        }

        $bank->delete();

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank berhasil dihapus.');
    }

    /**
     * Sync bank list from Moota.
     */
    public function syncFromMoota()
    {
        try {
            $mootaBanks = $this->mootaService->getAllBankAccounts();

            $synced = 0;
            foreach ($mootaBanks as $mootaBank) {
                Bank::updateOrCreate(
                    ['moota_bank_id' => $mootaBank['bank_id']],
                    [
                        'bank_name' => $mootaBank['label'] ?? $mootaBank['bank_type'] ?? 'Unknown',
                        'account_name' => $mootaBank['atas_nama'] ?? '',
                        'account_number' => $mootaBank['account_number'] ?? '',
                    ]
                );
                $synced++;
            }

            return back()->with('success', "Berhasil sync {$synced} bank dari Moota.");
        } catch (\Exception $e) {
            return back()->with('error', 'Gagal sync dari Moota: ' . $e->getMessage());
        }
    }
}
