<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Services\OrderService;
use App\Services\PaymentVerificationService;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function __construct(
        private OrderService $orderService,
        private PaymentVerificationService $verificationService,
    ) {
    }

    public function index(Request $request)
    {
        $query = Order::with(['storeLocation', 'bank']);

        // Filters
        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('order_number', 'like', "%{$request->search}%")
                    ->orWhere('customer_name', 'like', "%{$request->search}%")
                    ->orWhere('customer_whatsapp', 'like', "%{$request->search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('store_location_id')) {
            $query->where('store_location_id', $request->store_location_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('amount_min')) {
            $query->where('total_amount', '>=', $request->amount_min);
        }

        if ($request->filled('amount_max')) {
            $query->where('total_amount', '<=', $request->amount_max);
        }

        // Sorting
        $sortBy = $request->input('sort', 'created_at');
        $sortOrder = $request->input('order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $orders = $query->paginate(20);

        $storeLocations = \App\Models\StoreLocation::all();

        return view('admin.orders.index', compact('orders', 'storeLocations'));
    }

    public function show(Order $order)
    {
        $order->load(['items.product', 'storeLocation', 'bank']);

        return view('admin.orders.show', compact('order'));
    }

    /**
     * Update order status to confirm.
     */
    public function confirm(Order $order)
    {
        if (!$order->isPending()) {
            return back()->with('error', 'Order tidak dalam status pending.');
        }

        $order->markAsConfirmed();

        return back()->with('success', 'Order berhasil dikonfirmasi.');
    }

    /**
     * Update order status to complete.
     */
    public function complete(Order $order)
    {
        if (!$order->isConfirmed()) {
            return back()->with('error', 'Order harus dikonfirmasi terlebih dahulu.');
        }

        try {
            $this->orderService->completeOrder($order);
            return back()->with('success', 'Order berhasil diselesaikan.');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    /**
     * Cancel pending order.
     */
    public function cancel(Order $order)
    {
        if (!$order->isPending()) {
            return back()->with('error', 'Hanya order pending yang dapat dibatalkan.');
        }

        try {
            $this->orderService->cancelOrder($order);
            return redirect()->route('admin.orders.index')
                ->with('success', 'Order berhasil dibatalkan.');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    /**
     * Manual verify payment via Moota API.
     */
    public function verifyPayment(Order $order)
    {
        if (!$order->isPending()) {
            return back()->with('error', 'Order tidak dalam status pending.');
        }

        $result = $this->verificationService->verifyOrderPayment($order);

        if ($result['verified']) {
            return back()->with('success', 'Pembayaran berhasil diverifikasi dan dikonfirmasi.');
        }

        return back()->with('warning', 'Pembayaran belum ditemukan. ' . ($result['message'] ?? ''));
    }

    /**
     * Update pickup code for order.
     */
    public function updatePickupCode(Request $request, Order $order)
    {
        $request->validate([
            'pickup_code' => 'required|string|max:20',
        ]);

        $order->update([
            'pickup_code' => strtoupper($request->pickup_code),
        ]);

        return back()->with('success', 'Pickup code berhasil disimpan.');
    }

    /**
     * Export orders.
     */
    public function export(Request $request)
    {
        // Simple CSV export
        $query = Order::with(['storeLocation', 'bank']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $orders = $query->orderBy('created_at', 'desc')->get();

        $filename = 'orders_' . now()->format('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($orders) {
            $file = fopen('php://output', 'w');

            // Header
            fputcsv($file, [
                'Order Number',
                'Customer Name',
                'WhatsApp',
                'Store Location',
                'Subtotal',
                'PPN',
                'Unique Code',
                'Total Amount',
                'Status',
                'Created At',
                'Paid At',
            ]);

            foreach ($orders as $order) {
                fputcsv($file, [
                    $order->order_number,
                    $order->customer_name,
                    $order->customer_whatsapp,
                    $order->storeLocation->name ?? '-',
                    $order->subtotal,
                    $order->ppn,
                    $order->unique_code,
                    $order->total_amount,
                    $order->status,
                    $order->created_at->format('Y-m-d H:i:s'),
                    $order->paid_at?->format('Y-m-d H:i:s') ?? '-',
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
