<?php

namespace App\Http\Controllers;

use App\Services\PaymentVerificationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class MootaWebhookController extends Controller
{
    public function __construct(
        private PaymentVerificationService $verificationService,
    ) {
    }

    /**
     * Handle incoming Moota webhook.
     * Route: POST /api/webhooks/moota
     */
    public function handle(Request $request): JsonResponse
    {
        Log::info('Moota webhook received', [
            'ip' => $request->ip(),
        ]);

        // Verify webhook signature
        $secret = config('services.moota.webhook_secret');
        if ($secret && $secret !== 'your_webhook_secret_here') {
            $signature = $request->header('X-Moota-Signature')
                ?? $request->header('Signature');

            if (
                !$signature || !hash_equals(
                    hash_hmac('sha256', $request->getContent(), $secret),
                    $signature
                )
            ) {
                Log::warning('Moota webhook invalid signature', ['ip' => $request->ip()]);
                return response()->json(['message' => 'Invalid signature'], 403);
            }
        }

        $result = $this->verificationService->processWebhook($request->all());

        return response()->json([
            'message' => 'Webhook processed',
            'processed' => $result['processed'],
            'matched' => $result['matched'],
        ]);
    }

    /**
     * Manual verification endpoint for polling.
     * Route: POST /api/verify-payment/{orderNumber}
     */
    public function verifyPayment(Request $request, string $orderNumber): JsonResponse
    {
        $order = \App\Models\Order::where('order_number', $orderNumber)->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order tidak ditemukan.',
            ], 404);
        }

        $bankId = $request->input('bank_id') ?? $order->bank->moota_bank_id ?? null;
        $result = $this->verificationService->verifyOrderPayment($order, $bankId);

        return response()->json([
            'success' => $result['verified'],
            'message' => $result['message'],
            'mutation' => $result['mutation'] ?? null,
        ]);
    }

    /**
     * Batch verify all pending orders.
     * Route: POST /api/verify-pending-orders
     */
    public function verifyPendingOrders(Request $request): JsonResponse
    {
        $bankId = $request->input('bank_id');
        $result = $this->verificationService->verifyPendingOrders($bankId);

        return response()->json([
            'success' => true,
            'total' => $result['total'],
            'verified' => $result['verified'],
            'details' => $result['details'],
        ]);
    }
}
