<?php

namespace App\Jobs;

use App\Models\Order;
use App\Services\PaymentVerificationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class VerifyPaymentJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $backoff = 60; // Retry after 60 seconds

    public function __construct(
        public Order $order,
        public ?string $bankId = null,
    ) {
    }

    public function handle(PaymentVerificationService $verificationService): void
    {
        // Skip if order is no longer pending
        if (!$this->order->isPending()) {
            Log::info("Order {$this->order->order_number} is no longer pending, skipping verification");
            return;
        }

        $result = $verificationService->verifyOrderPayment($this->order, $this->bankId);

        Log::info("Payment verification for order {$this->order->order_number}", [
            'verified' => $result['verified'],
            'message' => $result['message'],
        ]);
    }
}
