<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Order extends Model
{
    use HasFactory;

    public const STATUS_PENDING = 'pending';
    public const STATUS_CONFIRM = 'confirm';
    public const STATUS_COMPLETE = 'complete';

    protected $fillable = [
        'order_number',
        'customer_name',
        'customer_whatsapp',
        'customer_address',
        'notes',
        'store_location_id',
        'bank_id',
        'subtotal',
        'ppn',
        'unique_code',
        'pickup_code',
        'payment_proof',
        'total_amount',
        'status',
        'expired_at',
        'paid_at',
        'completed_at',
        'moota_mutation_id',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'ppn' => 'decimal:2',
        'unique_code' => 'integer',
        'total_amount' => 'decimal:2',
        'expired_at' => 'datetime',
        'paid_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    // Relationships
    public function storeLocation(): BelongsTo
    {
        return $this->belongsTo(StoreLocation::class);
    }

    public function bank(): BelongsTo
    {
        return $this->belongsTo(Bank::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    public function uniqueCodeRecord(): HasOne
    {
        return $this->hasOne(UniqueCode::class);
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    public function scopeConfirmed($query)
    {
        return $query->where('status', self::STATUS_CONFIRM);
    }

    public function scopeComplete($query)
    {
        return $query->where('status', self::STATUS_COMPLETE);
    }

    public function scopeNotExpired($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('expired_at')
                ->orWhere('expired_at', '>', now());
        });
    }

    public function scopeExpired($query)
    {
        return $query->whereNotNull('expired_at')
            ->where('expired_at', '<=', now());
    }

    public function scopeByAmount($query, float $amount)
    {
        return $query->where('total_amount', $amount);
    }

    // Status helpers
    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isConfirmed(): bool
    {
        return $this->status === self::STATUS_CONFIRM;
    }

    public function isComplete(): bool
    {
        return $this->status === self::STATUS_COMPLETE;
    }

    public function isExpired(): bool
    {
        return $this->expired_at && $this->expired_at->isPast();
    }

    // Status transitions
    public function markAsConfirmed(?string $mutationId = null): bool
    {
        if (!$this->isPending()) {
            return false;
        }

        $this->update([
            'status' => self::STATUS_CONFIRM,
            'paid_at' => now(),
            'moota_mutation_id' => $mutationId,
        ]);

        return true;
    }

    public function markAsComplete(): bool
    {
        if (!$this->isConfirmed()) {
            return false;
        }

        $this->update([
            'status' => self::STATUS_COMPLETE,
            'completed_at' => now(),
        ]);

        return true;
    }

    // Generate order number
    public static function generateOrderNumber(): string
    {
        $prefix = 'FL';
        $date = now()->format('ymd');
        $random = strtoupper(substr(uniqid(), -4));

        return "{$prefix}{$date}{$random}";
    }
}
