<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Support\Facades\DB;

class OrderService
{
    private const PPN_RATE = 0.11; // 11% PPN
    private const ORDER_EXPIRY_HOURS = 1;

    public function __construct(
        private UniqueCodeService $uniqueCodeService,
    ) {
    }

    /**
     * Create a new order with items.
     *
     * @param array $customerData ['name', 'whatsapp', 'address', 'notes']
     * @param array $items [['product_id' => int, 'quantity' => int], ...]
     * @param int $storeLocationId
     * @param int $bankId
     * @return Order
     * @throws \Exception
     */
    public function createOrder(
        array $customerData,
        array $items,
        int $storeLocationId,
        int $bankId,
    ): Order {
        return DB::transaction(function () use ($customerData, $items, $storeLocationId, $bankId) {
            // Calculate subtotal and validate stock
            $subtotal = 0;
            $orderItems = [];

            foreach ($items as $item) {
                $product = Product::lockForUpdate()->findOrFail($item['product_id']);

                if (!$product->is_active) {
                    throw new \Exception("Produk {$product->name} tidak tersedia.");
                }

                if (!$product->decrementStock($item['quantity'])) {
                    throw new \Exception("Stok {$product->name} tidak mencukupi.");
                }

                $itemSubtotal = $product->price * $item['quantity'];
                $subtotal += $itemSubtotal;

                $orderItems[] = [
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'quantity' => $item['quantity'],
                    'unit_price' => $product->price,
                    'subtotal' => $itemSubtotal,
                ];
            }

            // Calculate PPN and unique code
            $ppn = $subtotal * self::PPN_RATE;
            $baseAmount = $subtotal + $ppn;
            $uniqueCode = $this->uniqueCodeService->generate($baseAmount);
            $totalAmount = $baseAmount + $uniqueCode;

            // Create order
            $order = Order::create([
                'order_number' => Order::generateOrderNumber(),
                'customer_name' => $customerData['name'],
                'customer_whatsapp' => $customerData['whatsapp'],
                'customer_address' => $customerData['address'],
                'notes' => $customerData['notes'] ?? null,
                'store_location_id' => $storeLocationId,
                'bank_id' => $bankId,
                'subtotal' => $subtotal,
                'ppn' => $ppn,
                'unique_code' => $uniqueCode,
                'total_amount' => $totalAmount,
                'status' => Order::STATUS_PENDING,
                'expired_at' => now()->addHours(self::ORDER_EXPIRY_HOURS),
            ]);

            // Create order items
            foreach ($orderItems as $itemData) {
                $order->items()->create($itemData);
            }

            // Mark unique code as used
            $this->uniqueCodeService->markAsUsed($uniqueCode, $baseAmount, $order->id);

            return $order->load(['items', 'storeLocation', 'bank']);
        });
    }

    /**
     * Cancel an order and restore stock.
     */
    public function cancelOrder(Order $order): void
    {
        if (!$order->isPending()) {
            throw new \Exception('Only pending orders can be cancelled.');
        }

        DB::transaction(function () use ($order) {
            // Restore stock
            foreach ($order->items as $item) {
                $product = Product::find($item->product_id);
                if ($product) {
                    $product->incrementStock($item->quantity);
                }
            }

            // Release unique code
            $baseAmount = $order->subtotal + $order->ppn;
            $this->uniqueCodeService->release($order->unique_code, $baseAmount);

            // Delete order
            $order->delete();
        });
    }

    /**
     * Complete an order after confirmation.
     */
    public function completeOrder(Order $order): void
    {
        if (!$order->isConfirmed()) {
            throw new \Exception('Only confirmed orders can be completed.');
        }

        $order->markAsComplete();

        // Release unique code for reuse
        $baseAmount = $order->subtotal + $order->ppn;
        $this->uniqueCodeService->release($order->unique_code, $baseAmount);
    }

    /**
     * Expire old pending orders.
     */
    public function expireOldOrders(): int
    {
        $expiredOrders = Order::pending()
            ->expired()
            ->get();

        $count = 0;
        foreach ($expiredOrders as $order) {
            try {
                $this->cancelOrder($order);
                $count++;
            } catch (\Exception $e) {
                // Log error but continue
                \Log::error("Failed to expire order {$order->id}: " . $e->getMessage());
            }
        }

        return $count;
    }
}
